<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

class MigrateLabTestsToProducts extends Migration
{
    public function up()
    {
        // 1. Add columns to products
        Schema::table('products', function (Blueprint $table) {
            if (!Schema::hasColumn('products', 'parent_id')) {
                $table->unsignedInteger('parent_id')->nullable()->after('category_id');
            }
            if (!Schema::hasColumn('products', 'is_storable')) {
                $table->tinyInteger('is_storable')->default(1)->after('quantity');
            }
            if (!Schema::hasColumn('products', 'normal_range')) {
                $table->string('normal_range', 50)->nullable()->after('inter_price');
            }
            if (!Schema::hasColumn('products', 'description')) {
                $table->string('description', 255)->nullable()->after('normal_range');
            }
            if (!Schema::hasColumn('products', 'test_category')) {
                $table->string('test_category', 50)->nullable()->after('description')->comment('Lab test type: Blood Test, X-Ray, etc');
            }
        });

        $idMap = []; // old lab_tests_catalog id => new product id

        if (Schema::hasTable('lab_tests_catalog')) {
            // 2a. Migrate from existing lab_tests_catalog
            $catalog = DB::table('lab_tests_catalog')->orderByRaw('COALESCE(parent_id, 999999), id')->get();
            foreach ($catalog as $row) {
                $parentProductId = isset($idMap[$row->parent_id]) ? $idMap[$row->parent_id] : null;
                $productId = DB::table('products')->insertGetId([
                    'company_id'    => $row->company_id,
                    'name'          => $row->name,
                    'category_id'   => 1,
                    'parent_id'     => $parentProductId,
                    'private_price' => (float) ($row->price ?? 0),
                    'quantity'      => 0,
                    'is_storable'   => 0,
                    'normal_range'  => $row->normal_range,
                    'description'   => $row->description,
                    'test_category' => $row->category ?? null,
                    'status'        => $row->is_active ?? 1,
                    'created_by'    => $row->created_by,
                    'created_at'    => $row->created_at,
                    'updated_at'    => $row->updated_at ?? now(),
                ]);
                $idMap[$row->id] = $productId;
            }
            Schema::dropIfExists('lab_tests_catalog');
        } else {
            // 2b. lab_tests_catalog already dropped: insert known tests and map IDs
            $labCategoryId = DB::table('product_categories')->where('name', 'lab')->value('id') ?? 1;
            $now = now();

            $tests = [
                ['id' => 1, 'name' => 'testing', 'parent' => null, 'price' => 10, 'normal_range' => '120/100', 'desc' => 'testing', 'cat' => 'Blood Test'],
                ['id' => 3, 'name' => 'Blood Test', 'parent' => null, 'price' => 50, 'normal_range' => '70-100', 'desc' => 'testing', 'cat' => 'X-Ray'],
                ['id' => 2, 'name' => 'hello', 'parent' => 1, 'price' => 20, 'normal_range' => '120/80', 'desc' => 'hello test', 'cat' => 'X-Ray'],
                ['id' => 4, 'name' => 'wbc', 'parent' => 1, 'price' => 2000, 'normal_range' => '5.51', 'desc' => null, 'cat' => 'Blood Test'],
            ];
            foreach ($tests as $t) {
                $parentProductId = isset($idMap[$t['parent']]) ? $idMap[$t['parent']] : null;
                $productId = DB::table('products')->insertGetId([
                    'company_id'    => 1,
                    'name'          => $t['name'],
                    'category_id'   => $labCategoryId,
                    'parent_id'     => $parentProductId,
                    'private_price' => $t['price'],
                    'quantity'      => 0,
                    'is_storable'   => 0,
                    'normal_range'  => $t['normal_range'],
                    'description'   => $t['desc'],
                    'test_category' => $t['cat'],
                    'status'        => 1,
                    'created_by'    => 1,
                    'created_at'    => $now,
                    'updated_at'    => $now,
                ]);
                $idMap[$t['id']] = $productId;
            }
        }

        // 3. Update lab_request_items.test_id to new product ids
        foreach ($idMap as $oldId => $newId) {
            DB::table('lab_request_items')->where('test_id', $oldId)->update(['test_id' => $newId]);
        }
    }

    public function down()
    {
        $cols = [];
        if (Schema::hasColumn('products', 'test_category')) $cols[] = 'test_category';
        if (Schema::hasColumn('products', 'description')) $cols[] = 'description';
        if (Schema::hasColumn('products', 'normal_range')) $cols[] = 'normal_range';
        if (Schema::hasColumn('products', 'is_storable')) $cols[] = 'is_storable';
        if (Schema::hasColumn('products', 'parent_id')) $cols[] = 'parent_id';
        if (!empty($cols)) {
            Schema::table('products', fn (Blueprint $t) => $t->dropColumn($cols));
        }
    }
}
