<?php
namespace App\Models\Stock;

use App\Models\User;
use Illuminate\Support\Facades\DB;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Sale extends Model
{
    use SoftDeletes;
  
    protected $fillable = [
        'type',
        'origin_ref',
        'reference',	
        'code',	
        'committed_date',	
        'total_amount',	
        'discounted_total',	
        'create_user',	
        'comment',	
        'amount_paid',
        'amount_remain',	
        'discount_perc',
        'discount_amount',
        'payment_date',
        'client_id',	
        'paid',
        'branch_id'
    ];

    protected $casts = [
        'committed_date' => 'date',
        'payment_date'   => 'date',
    ];

    protected $appends = ['payments'];

    protected $hidden = ['payments'];
    
    /**
     * @return BelongsTo
     */
    public function client()
    {
        return $this->belongsTo(Client::class, "client_id", 'id')
                    ->select('id', 'name', 'tinnumber', 'phone', 'address')
                    ->withTrashed();
    }


    /**
     * @return BelongsTo
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'create_user', 'id')
                    ->select('id', 'name','first_name','last_name')
                    ->withTrashed();
    }

    /**
     * Payment
     * return the first payment -  This is required for edit only
     * @return Illuminate\Support\Collection
     */
     public function getPaymentsAttribute()
     {
        return Payment::where('transaction_id', $this->id)
                        ->select('payment_type', DB::raw('SUM(amount_paid) as total_amount'), DB::raw('COUNT(payment_type) as mode_count'))
                        ->groupBy('payment_type')
                        ->get();
     }

     /**
     * @return BelongsTo
     */
    public function items()
    {
        return $this->hasMany(SaleItem::class, "sale_id", "id")
                    ->select('sale_items.*', 'products.name', 'products.code', 'stockin_histories.expiration_date')
                    ->join('products', 'sale_items.item_id', '=', 'products.id')
                    ->leftJoin('stockin_histories', 'stockin_histories.product_id', '=', 'products.id');
    }
}