<?php

namespace App\Models\Stock;

use App\Models\User;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Client extends Model
{
    use SoftDeletes;

    protected $table = 'clients';

    protected $fillable = [
        'name',
        'tinnumber',
        'phone',
        'email',
        'discount',
        'insurance_company',
        'policy_number',
        'coverage_percentage',
        'address',
        'sex',
        'date_of_birth',
    ];

    protected $casts = [
        'discount' => 'float',
        'coverage_percentage' => 'float',
        'date_of_birth' => 'date',
        'has_insurance' => 'boolean',
    ];

    protected $dates = [
        'date_of_birth',
        'created_at',
        'updated_at',
        'deleted_at'
    ];

    // Mutators for automatic has_insurance flag
    public function setInsuranceCompanyAttribute($value)
    {
        $this->attributes['insurance_company'] = $value;
        // Auto-set has_insurance based on insurance company
        $this->attributes['has_insurance'] = !empty($value);
    }

    // Accessor for has_insurance (computed)
    public function getHasInsuranceAttribute()
    {
        return !empty($this->insurance_company);
    }

    // Validation rules for the model
    public static function validationRules($id = null)
    {
        return [
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:50',
            'email' => 'nullable|email|max:50|unique:clients,email,' . $id,
            'discount' => 'nullable|numeric|between:0,100',
            'insurance_company' => 'nullable|string|max:255',
            'policy_number' => 'nullable|string|max:255',
            'coverage_percentage' => 'nullable|numeric|between:0,100',
            'sex' => 'nullable|in:M,F,other',
            'date_of_birth' => 'nullable|date|before:today',
            'address' => 'nullable|string|max:255',
        ];
    }

    // Scopes
    public function scopeHasInsurance($query)
    {
        return $query->whereNotNull('insurance_company')
                     ->where('insurance_company', '!=', '');
    }

    public function scopeByInsuranceCompany($query, $company)
    {
        return $query->where('insurance_company', $company);
    }

    public function scopeHighCoverage($query, $percentage = 80)
    {
        return $query->where('coverage_percentage', '>=', $percentage);
    }

    // Helper methods
    public function getFormattedDiscountAttribute()
    {
        return number_format($this->discount, 2) . '%';
    }

    public function getFormattedCoverageAttribute()
    {
        return number_format($this->coverage_percentage, 2) . '%';
    }

    public function getAgeAttribute()
    {
        if (!$this->date_of_birth) {
            return null;
        }
        
        return $this->date_of_birth->age;
    }
}