<?php
// app/Models/LabTestCatalog.php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class LabTestCatalog extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'products';

    protected $fillable = [
        'company_id',
        'product_id',
        'parent_id',
        'name',
        'category',
        'price',
        'normal_range',
        'description',
        'is_active',
        'created_by'
    ];

    protected $casts = [
        'price' => 'decimal:2',
        'is_active' => 'boolean',
    ];

    protected $appends = ['children_total'];

    /**
     * Get the company that owns the test.
     */
    public function company(): BelongsTo
    {
        return $this->belongsTo(Company::class);
    }

    /**
     * Linked product (service) used for invoicing.
     */
    public function product(): BelongsTo
    {
        return $this->belongsTo(\App\Models\Stock\Product::class, 'product_id');
    }

    /**
     * Get the parent test.
     */
    public function parent(): BelongsTo
    {
        return $this->belongsTo(LabTestCatalog::class, 'parent_id');
    }

    /**
     * Get the child tests.
     */
    public function children(): HasMany
    {
        return $this->hasMany(LabTestCatalog::class, 'parent_id');
    }

    /**
     * Get the user who created the test.
     */
    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Get active tests.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Get tests by category.
     */
    public function scopeByCategory($query, $category)
    {
        return $query->where('category', $category);
    }

    /**
     * Get main categories (without parent).
     */
    public function scopeMainCategories($query)
    {
        return $query->whereNull('parent_id');
    }

    /**
     * Get sub categories.
     */
    public function scopeSubCategories($query)
    {
        return $query->whereNotNull('parent_id');
    }

    /**
     * Search by name or description.
     */
    public function scopeSearch($query, $search)
    {
        return $query->where(function ($q) use ($search) {
            $q->where('name', 'like', "%{$search}%")
              ->orWhere('description', 'like', "%{$search}%");
        });
    }

    /**
     * Get children count attribute.
     */
    public function getChildrenTotalAttribute(): int
    {
        return $this->children()->count();
    }

    /**
     * Format price with currency.
     */
    public function getFormattedPriceAttribute(): string
    {
        return $this->price ? '$' . number_format($this->price, 2) : 'N/A';
    }

    /**
     * Check if test has children.
     */
    public function hasChildren(): bool
    {
        return $this->children()->exists();
    }

    /**
     * Get full path including parent names.
     */
    public function getFullPathAttribute(): string
    {
        $path = [$this->name];
        $parent = $this->parent;
        
        while ($parent) {
            array_unshift($path, $parent->name);
            $parent = $parent->parent;
        }
        
        return implode(' > ', $path);
    }
}