<?php

namespace App\Http\Controllers\api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\Lab\LabRequest;
use App\Models\Lab\LabRequestItem;
use App\Models\Stock\Client;
use App\Models\Stock\Product;

class LabRequestsController extends Controller
{
    /**
     * List lab requests with basic patient info and statistics.
     */
    public function index(Request $request)
    {
        $query = LabRequest::with([
                'patient:id,name,phone,email,sex,date_of_birth',
                'sale:id,reference'
            ])
            ->orderByDesc('id');

        if ($request->filled('status')) {
            $query->where('status', $request->get('status'));
        }

        if ($request->filled('patient_id')) {
            $query->where('patient_id', $request->get('patient_id'));
        }

        if ($request->filled('search')) {
            $search = $request->get('search');
            $query->whereHas('patient', function ($q) use ($search) {
                $q->where('name', 'LIKE', "%{$search}%")
                  ->orWhere('phone', 'LIKE', "%{$search}%");
            });
        }

        $perPage = (int) $request->get('per_page', 25);

        return response()->json([
            'status' => 1,
            'rows'   => $query->paginate($perPage),
        ]);
    }

    /**
     * Show a single lab request with items and related data.
     */
    public function show($id)
    {
        $labRequest = LabRequest::with([
                'patient',
                'items.test',
                'sale',
            ])->findOrFail($id);

        $patient = $labRequest->patient;

        $tests = $labRequest->items->map(function (LabRequestItem $item) {
            $test = $item->test;

            return [
                'id'               => $item->id,
                'test_id'          => $item->test_id,
                'name'             => $test ? $test->name : null,
                'category'         => $test ? ($test->test_category ?? optional($test->category)->name) : null,
                'price'            => $item->price,
                'normal_range'     => $test ? ($test->normal_range ?? $item->reference_range) : $item->reference_range,
                'result_value'     => $item->result_value,
                'unit'             => $item->unit,
                'status'           => $item->status,
                'reference_range'  => $item->reference_range,
            ];
        })->values();

        $row = [
            'id'            => $labRequest->id,
            'request_id'    => $labRequest->id,
            'patient_id'    => $labRequest->patient_id,
            'patient_name'  => $patient ? $patient->name : null,
            'doctor_name'   => $labRequest->doctor_name,
            'priority'      => $labRequest->priority,
            'sample_type'   => $labRequest->sample_type,
            'notes'         => $labRequest->notes,
            'status'        => $labRequest->status,
            'total_amount'  => $labRequest->total_amount,
            'request_date'  => $labRequest->request_date,
            'created_at'    => $labRequest->created_at,
            'patient'       => $patient,
            'tests'         => $tests,
            'sale_reference'=> optional($labRequest->sale)->reference,
        ];

        return response()->json([
            'status' => 1,
            'row'    => $row,
        ]);
    }

    public function store(Request $request)
    {
        $request->validate([
            'patient_id'        => 'required|integer|exists:clients,id',
            'doctor_name'       => 'required|string|max:255',
            'request_date'      => 'nullable|date',
            'priority'          => 'nullable|in:low,normal,high,urgent',
            'sample_type'       => 'nullable|string|max:50',
            'notes'             => 'nullable|string',
            'tests'             => 'required|array|min:1',
            'tests.*.test_id'   => 'required|integer|exists:products,id',
            'tests.*.price'     => 'required|numeric|min:0'
        ]);

        DB::beginTransaction();

        try {
            // 1️⃣ Create lab request (temporary total_amount = 0, will be recalculated)
            $labRequest = LabRequest::create([
                'patient_id'   => $request->patient_id,
                'request_date' => $request->request_date ?: now()->toDateString(),
                'doctor_name'  => $request->doctor_name,
                'priority'     => $request->priority ?? 'normal',
                'sample_type'  => $request->sample_type,
                'notes'        => $request->notes,
                'status'       => 'pending',
                'created_by'   => auth()->id(),
                'total_amount' => 0
            ]);

            // 2️⃣ Insert requested tests, expanding panels into sub-tests (products with is_storable=0)
            $testIds = collect($request->tests)->pluck('test_id')->unique()->values();
            $catalogTests = Product::withoutGlobalScope(\App\Scopes\CompanyScope::class)
                ->where('is_storable', 0)
                ->with('children')
                ->whereIn('id', $testIds)
                ->get()
                ->keyBy('id');

            foreach ($request->tests as $test) {
                /** @var \App\Models\Stock\Product|null $product */
                $product = $catalogTests->get($test['test_id']);

                if ($product && $product->children->count() > 0) {
                    // Panel: create one item per sub-test (analyte)
                    foreach ($product->children as $child) {
                        LabRequestItem::create([
                            'lab_request_id' => $labRequest->id,
                            'test_id'        => $child->id,
                            'price'          => (float) ($child->private_price ?? 0),
                            'status'         => 'pending',
                        ]);
                    }
                } else {
                    // Simple test
                    LabRequestItem::create([
                        'lab_request_id' => $labRequest->id,
                        'test_id'        => $test['test_id'],
                        'price'          => isset($test['price']) ? (float) $test['price'] : ((float) optional($product)->private_price ?: 0),
                        'status'         => 'pending',
                    ]);
                }
            }

            // 3️⃣ Recalculate total amount from all items
            $totalAmount = $labRequest->items()->sum('price');
            $labRequest->total_amount = $totalAmount;
            $labRequest->save();

            DB::commit();

            return response()->json([
                'status'  => 1,
                'message' => 'Lab request created successfully',
                'data'    => $labRequest->load('items')
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'status'  => 0,
                'message' => 'Failed to create lab request',
                'error'   => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update request meta data and associated tests (doctor side).
     */
    public function update(Request $request, $id)
    {
        $labRequest = LabRequest::findOrFail($id);

        $request->validate([
            'patient_id'        => 'required|integer|exists:clients,id',
            'doctor_name'       => 'required|string|max:255',
            'request_date'      => 'nullable|date',
            'priority'          => 'nullable|in:low,normal,high,urgent',
            'sample_type'       => 'nullable|string|max:50',
            'notes'             => 'nullable|string',
            'tests'             => 'required|array|min:1',
            'tests.*.test_id'   => 'required|integer|exists:products,id',
            'tests.*.price'     => 'required|numeric|min:0'
        ]);

        DB::beginTransaction();

        try {
            $labRequest->update([
                'patient_id'   => $request->patient_id,
                'request_date' => $request->request_date ?: $labRequest->request_date,
                'doctor_name'  => $request->doctor_name,
                'priority'     => $request->priority ?? 'normal',
                'sample_type'  => $request->sample_type,
                'notes'        => $request->notes,
                'total_amount' => 0,
            ]);

            // Sync tests: delete existing items and recreate (panel-aware)
            $labRequest->items()->delete();

            $testIds = collect($request->tests)->pluck('test_id')->unique()->values();
            $catalogTests = Product::withoutGlobalScope(\App\Scopes\CompanyScope::class)
                ->where('is_storable', 0)
                ->with('children')
                ->whereIn('id', $testIds)
                ->get()
                ->keyBy('id');

            foreach ($request->tests as $test) {
                /** @var \App\Models\Stock\Product|null $product */
                $product = $catalogTests->get($test['test_id']);

                if ($product && $product->children->count() > 0) {
                    foreach ($product->children as $child) {
                        LabRequestItem::create([
                            'lab_request_id' => $labRequest->id,
                            'test_id'        => $child->id,
                            'price'          => (float) ($child->private_price ?? 0),
                            'status'         => 'pending',
                        ]);
                    }
                } else {
                    LabRequestItem::create([
                        'lab_request_id' => $labRequest->id,
                        'test_id'        => $test['test_id'],
                        'price'          => isset($test['price']) ? (float) $test['price'] : ((float) optional($product)->private_price ?: 0),
                        'status'         => 'pending',
                    ]);
                }
            }

            // Recalculate total amount from all items
            $totalAmount = $labRequest->items()->sum('price');
            $labRequest->total_amount = $totalAmount;
            $labRequest->save();

            DB::commit();

            return response()->json([
                'status'  => 1,
                'message' => 'Lab request updated successfully',
                'data'    => $labRequest->load('items')
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'status'  => 0,
                'message' => 'Failed to update lab request',
                'error'   => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Enter / update lab results for a request (lab side).
     */
    public function updateResults(Request $request, $id)
    {
        $labRequest = LabRequest::with('items')->findOrFail($id);

        $request->validate([
            'status'          => 'nullable|in:pending,sample_collected,testing,completed,validated',
            'items'           => 'required|array|min:1',
            'items.*.id'      => 'required|integer|exists:lab_request_items,id',
            'items.*.result_value'    => 'nullable|string|max:255',
            'items.*.unit'            => 'nullable|string|max:50',
            'items.*.reference_range' => 'nullable|string|max:100',
            'items.*.status'          => 'nullable|in:pending,in_progress,done,validated',
        ]);

        DB::beginTransaction();

        try {
            foreach ($request->items as $itemData) {
                /** @var LabRequestItem $item */
                $item = $labRequest->items->firstWhere('id', $itemData['id']);
                if (!$item) {
                    continue;
                }

                $item->update([
                    'result_value'    => $itemData['result_value'] ?? $item->result_value,
                    'unit'            => $itemData['unit'] ?? $item->unit,
                    'reference_range' => $itemData['reference_range'] ?? $item->reference_range,
                    'status'          => $itemData['status'] ?? $item->status,
                ]);
            }

            if ($request->filled('status')) {
                $labRequest->status = $request->get('status');
                $labRequest->save();
            }

            DB::commit();

            return response()->json([
                'status'  => 1,
                'message' => 'Lab results updated successfully',
                'data'    => $labRequest->fresh()->load(['patient', 'items.test']),
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'status'  => 0,
                'message' => 'Failed to update lab results',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }
}