<?php

namespace App\Http\Controllers\Stock;

use Carbon\Carbon;
use App\Models\Stock\Unit;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Models\Stock\Product;
use App\Imports\ProductImport;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Models\Stock\StockinHistory;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Facades\Excel;
use App\Models\Stock\ProductCategory;
use Illuminate\Support\Facades\Storage;
use App\Models\Stock\ProductCategory as Category;

class ProductsController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\JsonResponse
     */

     public function index()
     {
        return response()->json([
            'status'=>1,
            'rows'  => Product::where('is_storable', 1)->orderByDesc('id')->with('category', 'unit')->paginate(\request()->query('per_page') ?? 45)
        ]);
     }


     /**
      * Upload file
      */
      public function storeFile(Request $request)
      {
          $file = $request->file('file');
          $folder = 'product_images/';
          $id = Auth::id();
          if ($id) {
              $folder .= sprintf('%04d', (int)$id / 1000) . '/' . $id . '/';
          }
          $folder = $folder . date('Y/m/d');
          $newFileName = Str::slug(substr($file->getClientOriginalName(), 0, strrpos($file->getClientOriginalName(), '.')));
          if(empty($newFileName)) $newFileName = md5($file->getClientOriginalName());
  
          $i = 0;
          do {
              $newFileName2 = $newFileName . ($i ? $i : '');
              $testPath = $folder . '/' . $newFileName2 . '.' . $file->getClientOriginalExtension();
              $i++;
          } while (Storage::disk('public')->exists($testPath));
  
          $check = $file->storeAs( $folder, $newFileName2 . '.' . $file->getClientOriginalExtension(),'public');
          $product = Product::find($request->input('product_id'));
          
          $fullPath = Storage::path('public/'.$product->image);
          $pathToUnlink = str_replace('/','/',$fullPath);
          if(!empty($product->image)) {
            unlink($pathToUnlink); 
          }
          
          $product->image = $check;
          $product->save();
          return response()->json([
            'status'  => 1,
            'path'    => $check,
            'message' => 'Saved sucessfully'
          ]);
      }

      /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */

     public function store(Request $request)
     {
        // Check if request contain id then perfom update

        if($request->has('id')){
            $product = Product::find($request->input('id'));
        } else{
            $product = new Product();
            $product->reference = generateReference(20);
            $product->status = 1;
            $product->quantity = 0;
            $product->is_storable = 1;
            if (empty($request->input('code'))) { 
                $product->code = generateRowCode(8);
            }
        }
        
        $category = Category::where('id',$request->input('category_id'))
                              ->first();

        $product->fill($request->input());
        $product->pcode = createProductCode($category->name,$request->input('name'));
        $product->cost_price = $request->input('cost_price');
        $product->private_price = $request->input('private_price');
        $product->rhia_price = $request->input('rhia_price');
        $product->inter_price = $request->input('inter_price');
        $product->packaging = $request->input('packaging');
        $product->image = NULL;
        $product->save();
        $id = $product->id;

        return response()->json([
            'status' => 1,
            'row'    => Product::where('id', $product->id)->with('creator', 'company', 'category', 'unit')->first()
        ]);
     }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */

     public function storeImported(Request $request)
     {
        // Check if request contain id then perfom update

        foreach($request->input() as $index =>  $product){
            $data = json_decode($product, true);
            if (isset($data['name']) AND isset($data['category'])) {
                $product = new Product();
                $category = Category::where('id',$data['category'])
                              ->first();

                $product->reference = generateReference(20);
                $product->pcode = createProductCode($category->name, $data['name']);
                $product->status = 1;
                $product->code = generateRowCode(8);
                $product->name = $data['name'];
                $product->cost_price = $data['purchase_price'] ?? 0;
                $product->private_price = $data['selling_price'] ?? 0;
                $product->rhia_price = 0;
                $product->inter_price = 0;
                $product->quantity = 0;
                $product->unit_id = $data['unit'] ?? NULL;
                $product->category_id = $data['category'] ?? NULL;
                $product->packaging = $data['packaging'] ?? NULL;
                $product->image = NULL;
                $product->created_by = Auth::id();
                $product->save();
            }
        }

        return response()->json([
            'status' => 1,
            'message'    => "Products Imported Successfuly"
        ]);
     }

    /**
     * Import Products list(excel) into Products table.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */

     public function ImportProducts(Request $request){
        if ($request->hasFile('excel_file')) {
            
            $request->validate([
                'excel_file' => 'required|file|mimes:xls,xlsx',
            ]);
            
            $file = $request->file('excel_file');
            $import = new ProductImport();
            $UploadedData = Excel::import($import, $file);
            $importedData = $import->getImportedData();
            $getMissingColumns = $import->getMissingColumns();
            return response()->json([
                'importedData' => $importedData,
                'getMissingColumns' => $getMissingColumns,
                'message' => 'Excel file Imported successfully'
            ]);
        }

        return response()->json([
            'message' => 'No file uploaded'
        ], 400);
     }

    /**
     * Show Expired Products.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */

     public function expired(){
        $expired = 'EXPIRED';
        $expiredData = StockinHistory::where('status', $expired)
                                    ->with('product')->paginate(\request()->query('per_page') ?? 45);
        return response()->json([
            'status' => 1,
            'rows'  => $expiredData
        ]);
     }

    /**
     * Show Expired Products.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */

     public function expirationCheckInthreeMo(){
        $products = StockinHistory::whereDate('expiration_date', '<=', Carbon::now()->addMonths(3))
                            ->whereNotIn('status', ['EXPIRED','CONSUMED'])   
                            ->with('product') 
                            ->orderBy('expiration_date', 'ASC')             
                            ->get();

        if ($products->isNotEmpty()) {
            return response()->json([
                'message' => "expired products",
                'products' => $products
            ]);
        }
        
        return response()->json([
                'message' => "No expired products found",
                'products' => []
        ]);
     }

     /**
     * Display the specified resource.
     *
     * @param  int $id
     * @return \Illuminate\Http\JsonResponse
     */

     public function show($id)
     {
        $product = Product::findOrFail($id);
        if(!$product){
            return response()->json([
                'status'=> 0,
                'error' => 'Product can\'t Found!'
            ]);
        }

        return response()->json([
            'status'=> 1,
            'row'   => $product
        ]);
     }


    /**
     * Remove the specified resource from storage.
     *
     * @param  int $is
     * @return \Illuminate\Http\JsonResponse
     */

     public function destroy($id){
        $product = Product::findOrFail($id);
        if(!$product){
            return response()->json([
                'status'=>0,
                'error' =>'Product can\'t Found!'
            ]);
        }

        $product ->delete();

        return response()->json([
            'status'=>1,
            'message'=>'Product deleted Successfuly!'
        ]);
     } 

     /**
     * search fora category
     * @param Request $request
     * @return JsonResponse
     */
    public function search(Request $request)
    {
        if (empty($department = $request->input('department'))) {
            $result = Product::select('products.id', 'products.name', 'code', 'cost_price', 'quantity', 'rhia_price', 'private_price','inter_price','packaging', 'units.name as unit')
                          ->leftJoin('units', 'products.unit_id', '=', 'units.id');
            if (!empty($request->get('with_quantity'))) {
                $result->where('quantity', '>', 0);
            }
        } else {
            $result = Product::select('products.id', 'products.name', 'code', 'cost_price', 'stock.quantity', 'rhia_price', 'private_price','inter_price','packaging', 'units.name as unit')
                          ->leftJoin('units', 'products.unit_id', '=', 'units.id')
                          ->leftJoin('stock', function($join) use ($department){
                            $join->on('products.id', '=', 'stock.product_id');
                            $join->on('stock.department_id', '=', DB::raw("{$department}"));
                          });
            if (!empty($request->get('with_quantity'))) {
                $result->where('stock.quantity', '>', 0);
            }
        }
          
        $keyword = $request->get('query');
        if (empty($keyword)) {
            return  response()->json($result->with('ProductChanges')->orderBy('products.name', 'ASC')->take(2000)->get());
        } else {
            return response()->json($result->where('products.name', 'LIKE', '%' . $keyword . '%')->with('ProductChanges')->orderBy('name', 'ASC')->get());
        }
    }

    /**
     * Get products extraxs
     * @return JsonResponse
     */
    public function extras()
    {
        return response()->json([
            'status'     => 1,
            'categories' => ProductCategory::orderBy('name', 'ASC')->get(),
            'units'      => Unit::all()
        ]);
    }

    /**
    * 
    * Handle Bulk Delete
    * @param string $id
    * @return \Illuminate\Http\JsonResponse
    */
    public function bulkDelete($ids)
    {
        $products = explode(",", $ids);
        Product::whereIn('id', $products)->delete();
        
        return response()->json([
            'status' => 1,
            'message' => 'Products deleted Successfuly!'
        ]);
    }
}
