<?php

namespace App\Http\Controllers\Stock;

use App\Models\Stock\Client;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;

class ClientsController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        return response()->json([
            'status' => 1,
            'rows' => Client::orderByDesc('id')->paginate(\request()->query('per_page') ?? 45)
        ]);
    }
    
    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
 
    public function store(Request $request)
{
    // Validate the request data
    $request->validate([
        'name' => 'required|string|max:255',
        'phone' => 'required|string|max:50',
        'email' => 'nullable|email|max:50',
        'discount' => 'nullable|numeric|min:0|max:100',
        'insurance_company' => 'nullable|string|max:255',
        'policy_number' => 'nullable|string|max:255',
        'coverage_percentage' => 'nullable|numeric|min:0|max:100',
        'has_insurance' => 'nullable|boolean',
        'sex' => 'nullable|in:M,F,other',
        'date_of_birth' => 'nullable|date',
        'address' => 'nullable|string|max:50',
    ]);

    // Find or create client
    if ($request->has('id') && $request->input('id') > 0) {
        $client = Client::find($request->input('id'));
        if (!$client) {
            return response()->json([
                'status' => 0,
                'message' => 'Client not found'
            ], 404);
        }
        $message = "Client Info Updated Successfully";
    } else {
        $client = new Client();
        $message = "New Client Created Successfully";
    }

    // Assign basic information
    $client->name = $request->input('name');
    $client->phone = $request->input('phone');
    $client->email = $request->input('email');
    $client->sex = $request->input('sex');
    $client->date_of_birth = $request->input('date_of_birth');
    
    // Handle discount - use null coalescing operator
    $discount = $request->input('discount');
    $client->discount = $discount !== null ? min(max($discount, 0), 100) : 0;
    
    $client->address = $request->input('address');
    
    // Handle insurance information
    $hasInsurance = $request->input('has_insurance', false);
    $insuranceCompany = $request->input('insurance_company');
    
    if ($hasInsurance && $insuranceCompany) {
        $client->insurance_company = $insuranceCompany;
        $client->policy_number = $request->input('policy_number');
        
        // Handle coverage percentage with validation
        $coverage = $request->input('coverage_percentage');
        $client->coverage_percentage = $coverage !== null ? min(max($coverage, 0), 100) : 0;
    } else {
        // Clear insurance fields if no insurance
        $client->insurance_company = null;
        $client->policy_number = null;
        $client->coverage_percentage = 0;
    }

    // Save the client
    $client->save();

    return response()->json([
        'status' => 1,
        'row' => Client::with([])->find($client->id), // You can add relationships if needed
        'message' => $message
    ]);
}

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
       $client = Client::findOrFail($id);
       if(!$client){
            return response()->json([
                'status' => 0,
                'error' => "Client not Found!"
            ]);
       }

       return response()->json([
            'status' => 1,
            'row' => $client
       ]);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $client = Client::findOrFail($id);
        if(!$client){
            return response()->json([
                'status' => 0,
                'error' => "Client not Found!"
            ]);
       }

       $client->delete();

        return response()->json([
            'status' => 1,
            'row' => "Customer Deleted Successfully!"
        ]);
    }


        /**
     * search fora category
     * @param Request $request
     * @return JsonResponse
     */
    public function search(Request $request)
    {
        $result = Client::select('id', 'name');
        $keyword = $request->get('query');
        if (empty($keyword)) {
            return  response()->json($result->orderBy('name', 'ASC')->take(250)->get());
        } else {
            return response()->json($result->where('name', 'LIKE', '%' . $keyword . '%')->orderBy('name', 'ASC')->get());
        }
    }
}
